/* m17n.h -- header file for the SHELL API of the m17n library.
   Copyright (C) 2003, 2004
     National Institute of Advanced Industrial Science and Technology (AIST)
     Registration Number H15PRO112

   This file is part of the m17n library.

   The m17n library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   The m17n library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the m17n library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.  */

#ifndef _M17N_H_
#define _M17N_H_

#include <stdio.h>
#include <stdlib.h>
#include <time.h>

#ifndef _M17N_CORE_H_
#include "m17n-core.h"
#endif

#ifdef __cplusplus
extern "C"
{
#endif

#if !defined (FOR_DOXYGEN) || defined (DOXYGEN_INTERNAL_MODULE)
extern void m17n_init (void);
#undef M17N_INIT
#define M17N_INIT() m17n_init ()

extern void m17n_fini (void);
#undef M17N_FINI
#define M17N_FINI() m17n_fini ()
#endif

/***en @defgroup m17nShell SHELL API */
/***ja @defgroup m17nShell VF API */
/*=*/

/*
 *  (11) Functions related to the m17n database
 */
/*=*/

/*** @ingroup m17nShell */
/***en @defgroup m17nDatabase Database */
/***ja @defgroup m17nDatabase f[^x[X */
/*=*/

/* Directory of an application specific databases.  */
extern char *mdatabase_dir;
/*=*/
/***
    @ingroup m17nDatabase  */ 
/***en
    @brief Type of database.

    The type #MDatabase is for a database object.  Its internal
    structure is concealed from an application program.  */
/***ja 
    @brief f[^x[X̌^錾.

    #MDatabase ^̓f[^x[XIuWFNgp̍\̂łB\
    ̓AvP[VvO͌ȂB
    */

typedef struct MDatabase MDatabase;

/*=*/

/* Look for a data.  */
extern MDatabase *mdatabase_find (MSymbol tag1, MSymbol tag2,
				  MSymbol tag3, MSymbol tag4);

extern MPlist *mdatabase_list (MSymbol tag0, MSymbol tag1,
			       MSymbol tag2, MSymbol tag3);

/* Load a data.  */
void *mdatabase_load (MDatabase *mdb);

/* Get tags of a data.  */
extern MSymbol *mdatabase_tag (MDatabase *mdb);

/* Define a data.  */
extern MDatabase *mdatabase_define (MSymbol tag1, MSymbol tag2,
				    MSymbol tag3, MSymbol tag4,
				    void *(*loader) (MSymbol *, void *),
				    void *extra_info);

/*=*/
/* (S2) Charset staffs */
/*=*/

/*** @ingroup m17nShell */
/***en @defgroup m17nCharset Charset */
/***ja @defgroup m17nCharset Zbg */
/*=*/

#if !defined (FOR_DOXYGEN) || defined (DOXYGEN_INTERNAL_MODULE)
#define MCHAR_INVALID_CODE 0xFFFFFFFF
#endif

/* Predefined charsets */ 
extern MSymbol Mcharset_ascii;
extern MSymbol Mcharset_iso_8859_1;
extern MSymbol Mcharset_unicode;
extern MSymbol Mcharset_m17n;
extern MSymbol Mcharset_binary;

/* Predefined keys for mchar_define_charset ().  */ 
extern MSymbol Mmethod;
extern MSymbol Mdimension;
extern MSymbol Mmin_range;
extern MSymbol Mmax_range;
extern MSymbol Mmin_code;
extern MSymbol Mmax_code;
extern MSymbol Mascii_compatible;
extern MSymbol Mfinal_byte;
extern MSymbol Mrevision;
extern MSymbol Mmin_char;
extern MSymbol Mmapfile;
extern MSymbol Mparents;
extern MSymbol Msubset_offset;
extern MSymbol Mdefine_coding;
extern MSymbol Maliases;

/* Methods of a charset.  */
extern MSymbol Moffset;
extern MSymbol Mmap;
extern MSymbol Munify;
extern MSymbol Msubset;
extern MSymbol Msuperset;

/* etc. */
extern MSymbol Mcharset;

extern MSymbol mchar_define_charset (const char *name, MPlist *plist);

extern MSymbol mchar_resolve_charset (MSymbol symbol);

extern int mchar_list_charset (MSymbol **symbols);

extern int mchar_decode (MSymbol charset_name, unsigned code);

extern unsigned mchar_encode (MSymbol charset_name, int c);

extern int mchar_map_charset (MSymbol charset_name,
			      void (*func) (int from, int to, void *arg),
			      void *func_arg);

/*=*/

/* (S3) code conversion */
/*=*/
/*** @ingroup m17nShell */
/***en @defgroup m17nConv Code Conversion */
/***ja @defgroup m17nConv R[hϊ */
/*=*/

/* Predefined coding systems */
extern MSymbol Mcoding_us_ascii;
extern MSymbol Mcoding_iso_8859_1;
extern MSymbol Mcoding_utf_8;
extern MSymbol Mcoding_utf_8_full;
extern MSymbol Mcoding_utf_16;
extern MSymbol Mcoding_utf_16be;
extern MSymbol Mcoding_utf_16le;
extern MSymbol Mcoding_utf_32;
extern MSymbol Mcoding_utf_32be;
extern MSymbol Mcoding_utf_32le;
extern MSymbol Mcoding_sjis;

/* Parameter keys for mconv_define_coding ().  */
extern MSymbol Mtype;
extern MSymbol Mcharsets;
extern MSymbol Mflags;
extern MSymbol Mdesignation;
extern MSymbol Minvocation;
extern MSymbol Mcode_unit;
extern MSymbol Mbom;
extern MSymbol Mlittle_endian;

/* Symbols representing coding system type.  */
extern MSymbol Mutf;
extern MSymbol Miso_2022;

/* Symbols appearing in the value of Mfrag parameter.  */
extern MSymbol Mreset_at_eol;
extern MSymbol Mreset_at_cntl;
extern MSymbol Meight_bit;
extern MSymbol Mlong_form;
extern MSymbol Mdesignation_g0;
extern MSymbol Mdesignation_g1;
extern MSymbol Mdesignation_ctext;
extern MSymbol Mdesignation_ctext_ext;
extern MSymbol Mlocking_shift;
extern MSymbol Msingle_shift;
extern MSymbol Msingle_shift_7;
extern MSymbol Meuc_tw_shift;
extern MSymbol Miso_6429;
extern MSymbol Mrevision_number;
extern MSymbol Mfull_support;

/* etc */
extern MSymbol Mcoding;
extern MSymbol Mmaybe;

/*** @ingroup m17nConv */
/***en
    @brief Codes that represent the result of code conversion.

    One of these values is set in @c MConverter-\>result.   */

/***ja
    @brief R[hϊ̌ʂR[h.

    ̒l̂ @c MConverter-\>result ɐݒ肳B  */

enum MConversionResult
  {
    /***en Code conversion is successful. */
    /***ja R[hϊ͐. */
    MCONVERSION_RESULT_SUCCESS,

    /***en On decoding, the source contains an invalid byte. */
    /***ja fR[hA\[XɕsȃoCg܂܂Ă. */
    MCONVERSION_RESULT_INVALID_BYTE,

    /***en On encoding, the source contains a character that cannot be
	encoded by the specified coding system. */
    /***ja GR[hAw̃R[hnŃGR[hłȂ\[
	XɊ܂܂Ă. */
    MCONVERSION_RESULT_INVALID_CHAR,

    /***en On decoding, the source ends with an incomplete byte sequence. */
    /***ja fR[hAsSȃoCgŃ\[XIĂB*/
    MCONVERSION_RESULT_INSUFFICIENT_SRC,

    /***en On encoding, the destination is too short to store the result. */
    /***ja GR[hAʂi[̈悪ZB */
    MCONVERSION_RESULT_INSUFFICIENT_DST,

    /***en An I/O error occurred in the conversion.  */
    /***ja R[hϊ I/O G[NB  */
    MCONVERSION_RESULT_IO_ERROR
  };
/*=*/

/*** @ingroup m17nConv */
/***en
    @brief Structure to be used in code conversion.

    Structure to be used in code conversion.  The first three members
    are to control the conversion.  */

/***ja
    @brief R[hϊɗp\.

    R[hϊɗp\́Bŏ̂R̃o͕ϊ𐧌䂷B

    @latexonly \IPAlabel{MConverter} @endlatexonly  
*/

typedef struct
{
  /***en
      Set the value to nonzero if the conversion should be lenient.
      By default, the conversion is strict (i.e. not lenient).

      If the conversion is strict, the converter stops at the first
      invalid byte (on decoding) or at the first character not
      supported by the coding system (on encoding).  If this happens,
      @c MConverter-\>result is set to @c
      MCONVERSION_RESULT_INVALID_BYTE or @c
      MCONVERSION_RESULT_INVALID_CHAR accordingly.

      If the conversion is lenient, on decoding, an invalid byte is
      kept per se, and on encoding, an invalid character is replaced
      with "<U+XXXX>" (if the character is a Unicode character) or
      with "<M+XXXXXX>" (otherwise).  */

  /***ja 
      ȕϊKvłȂꍇɒl 0 ȊOɂBftHgł́A
      ϊ͌łB

      ϊƂ́AfR[h̍ۂɂ͍ŏ̕sȃoCgŃRo[^
      ~܂邱ƁAGR[h̍ۂɂ͕ϊR[hnŃT|[g
      Ȃŏ̕ŃRo[^~܂邱ƂwB̏ꍇA
      @c MConverter-\>result ͂ꂼ @c
      MCONVERSION_RESULT_INVALID_BYTE @c
      MCONVERSION_RESULT_INVALID_CHAR ƂȂB

      ϊłȂꍇɂ́AfR[h̍ۂ̕sȃoCg͂̃oCg
      ̂܂܎cB܂GR[h̍ۂɂ́Asȕ Unicode 
      ꍇɂ "<U+XXXX>" ɁAłȂꍇɂ "<M+XXXXXX>" ɒu
      B  */

  int lenient;

  /***en
      Set the value to nonzero before decoding or encoding the last
      block of the byte sequence or the character sequence
      respectively.  The value influences the conversion as below.

      On decoding, in the case that the last few bytes are too short
      to form a valid byte sequence:

	If the value is nonzero, the conversion terminates by error
	(MCONVERSION_RESULT_INVALID_BYTE) at the first byte of the
	sequence.

	If the value is zero, the conversion terminates successfully.
	Those bytes are stored in the converter as carryover and are
	prepended to the byte sequence of the further conversion.

      On encoding, in the case that the coding system is context
      dependent:

	If the value is nonzero, the conversion may produce a byte
	sequence at the end to reset the context to the initial state
	even if the source characters are zero.

	If the value is zero, the conversion never produce such a byte
	sequence at the end.  */

/***ja 

     oCg̏I[̃ubNfR[hہA܂͕̏I[̃u
     bNGR[hۂ́Al 0 ȊOɂB̒l͈ȉ̂悤
     ɕϊɉeB

     fR[fBO̍ۂɍŌ̐oCgoCgV[NGX`
     ɂ͒ZꍇF

       l 0 łȂ΁Aῗ̃V[NGX̍ŏ̃oCgɂāA
       G[ (MCONVERSION_RESULT_INVALID_BYTE) ŏIB

       l 0 Ȃ΁Aϊ͐ďIB̐oCg̓L[I[
       oƂăRo[^ɕێA̕ϊsۂɕϊoC
       g̑OɕtB

     GR[fBO̍ۂɃR[hnˑ̏ꍇA

       l 0 łȂ΁AReLXgŏɖ߂߂̃oCg񂪃\[
       X̕ƂȂϊ̌ʐ邱ƂB

       l 0 Ȃ΁Â悤ȃoCg͐ȂB

	 */

  int last_block;

  /***en
      If the value is nonzero, it specifies at most how many
      characters to convert.  */
  /***ja
      0 łȂ΁Aϊő̕w肷B
      */

  unsigned at_most;

  /***en
      The following three members are to report the result of the
      conversion.  */
  /***ja
      ȉ̂R̃o͕ϊ̌ʂ񍐂B  */

  /***en
      Number of characters most recently decoded or encoded. */
  /***ja
      ŋ߂ɃfR[h/GR[hꂽB */

  int nchars;

  /***en
      Number of bytes recently decoded or encoded. */

  /***ja
      ŋ߂ɃfR[h/GR[hꂽoCgB */

  int nbytes;

  /***en
      Result code of the conversion. */

  /***ja
      R[hϊ̌ʂR[hB */

  enum MConversionResult result;

  /***en
      Various information about the status of code conversion.  The
      contents depend on the type of coding system.  It is assured
      that @c status is aligned so that any type of casting is safe
      and at least 256 bytes of memory space can be used.  */

  /***ja
      R[hϊ̏󋵂ɊւX̏Be̓R[hñ^Cvɂ
      ĈقȂB@c status ͂ǂ̂悤Ȍ^ւ̃LXgɑ΂ĂSȂ
      ɃACĂA܂Œ256oCg̃̈悪g
      悤ɂȂĂB  */

  union {
    void *ptr;
    double dbl;
    char c[256];
  } status;

  /***en
      This member is for internally use only.  An application program
      should never touch it.  */
  /***ja ̃o͓IɎgpAAvP[VvO͐G
      Ă͂ȂȂB */
  void *internal_info;
} MConverter;
/*=*/

/*** @ingroup m17nConv */
/***en 
    @brief Types of coding system.  */
/***ja 
    @brief R[hñ^Cv.  */

enum MCodingType
  {
    /***en
	A coding system of this type supports charsets directly.
	The dimension of each charset defines the length of bytes to
	represent a single character of the charset, and a byte
	sequence directly represents the code-point of a character.

	The m17n library provides the default decoding and encoding
	routines of this type.  */

    /***ja
	̃^CṽR[hn͕Zbg𒼐ڃT|[gBeZb
	g̎Ƃ́A̕Zbgňꕶ\邽߂ɕKvȃoC
	głAoCg͕̃R[h|Cg𒼐ڕ\킷B

	m17n Cu͂̃^Cvp̃ftHg̃GR[h^fR[
	h[eB񋟂B  */

    MCODING_TYPE_CHARSET,

    /***en
	A coding system of this type supports byte sequences of a
	UTF (UTF-8, UTF-16, UTF-32) like structure.

	The m17n library provides the default decoding and encoding
	routines of this type.  */

    /***ja
	̃^CṽR[hńAUTF n (UTF-8, UTF-16, UTF-32) ̃oC
	gT|[gB

	m17n Cu͂̃^Cvp̃ftHg̃GR[h^fR[
	h[eB񋟂B  */

    MCODING_TYPE_UTF,

    /***en
	A coding system of this type supports byte sequences of an
	ISO-2022 like structure.  The details of each structure are
	specified by @c MCodingInfoISO2022 .

	The m17n library provides decoding and encoding routines of
	this type.  */

    /***ja 
        ̃^CṽR[hńAISO-2022 ñoCgT|[gB
	eR[hn̍\̏ڍׂ @c MCodingInfoISO2022 Ŏw肳B

	m17n Cu͂̃^Cvp̃ftHg̃GR[h^fR[
	h[eB񋟂B  */

    MCODING_TYPE_ISO_2022,

    /***en
	A coding system of this type is for byte sequences of
	miscellaneous structures.

	The m17n library does not provide decoding and encoding
	routines of this type.  They must be provided by the
	application program.  */

    /***ja
	̃^CṽR[hńȂ̍\̃oCĝ߂̂̂ł
	B

	m17n Cu͂̃^Cvp̃GR[h^fR[h[eB
        񋟂Ȃ̂ŁAAvP[VvOŏ
        KvB  */

    MCODING_TYPE_MISC
  };
/*=*/

/*** @ingroup m17nConv */
/***en 
    @brief  Bit-masks to specify the detail of coding system whose type is MCODING_TYPE_ISO_2022.  */
/***ja 
    @brief MCODING_TYPE_ISO_2022 ^CṽR[hn̏ڍׂ\킷rbg}XN.  */

enum MCodingFlagISO2022
  {
    /***en
	On encoding, reset the invocation and designation status to
	initial at end of line.  */
    /***ja GR[h̍ہAsŌĂяo (invocation) Ǝw
	(designation) ̏Ԃlɖ߂B   */
    MCODING_ISO_RESET_AT_EOL =		0x1,

    /***en
	On encoding, reset the invocation and designation status to
	initial before any control codes.  */
    /***ja
	GR[h̍ہAׂĂ̐䕶̑OŁAĂяo
	(invocation) Ǝw (designation) ̏Ԃlɖ߂B        */
    MCODING_ISO_RESET_AT_CNTL =		0x2,

    /***en
	Use the right graphic plane.  */
    /***ja
	}`W̉EgB  */
    MCODING_ISO_EIGHT_BIT =		0x4,

    /***en
	Use the non-standard 4 bytes format for designation sequence
	for charsets JISX0208.1978, GB2312, and JISX0208.1983.  */
    /***ja
	JISX0208.1978, GB2312, JISX0208.1983 ̕Wɑ΂wV[
	NGXƂāAW4oCg`pB */

    MCODING_ISO_LONG_FORM =		0x8,

    /***en
	On encoding, unless explicitly specified, designate charsets
	to G0.  */
    /***ja 
        GR[h̍ہAɎw肳ȂAW G0 Ɏw
	B*/
    MCODING_ISO_DESIGNATION_G0 =		0x10,

    /***en
	On encoding, unless explicitly specified, designate charsets
	except for ASCII to G1.  */
    /***ja
	GR[h̍ہAɎw肳ȂAASCII ȊO̕W G1 
	ɎwB*/
    MCODING_ISO_DESIGNATION_G1 =		0x20,

    /***en
	On encoding, unless explicitly specified, designate 94-chars
	charsets to G0, 96-chars charsets to G1.  */
    /***ja
	GR[h̍ہAɎw肳ȂA94W G0 
	ɁA96W G1 ɎwB*/
    MCODING_ISO_DESIGNATION_CTEXT =	0x40,

    /***en
	On encoding, encode such charsets not conforming to ISO-2022
	by ESC % / ..., and encode non-supported Unicode characters by
	ESC % G ... ESC % @@ .  On decoding, handle those escape
	sequences.  */
    /***ja
	GR[h̍ہAISO-2022 ɍvȂW ESC % / ... ŃG
	R[hBT|[gĂȂ Unicode  ESC % G ...
	ESC % @@ ŃGR[hB
	fR[h̍ہÃGXP[vEV[PX߂B  */
    MCODING_ISO_DESIGNATION_CTEXT_EXT =	0x80,

    /***en
	Use locking shift.  */
    /***ja
	bLOVtggB  */
    MCODING_ISO_LOCKING_SHIFT =	0x100,

    /***en
	Use single shift (SS2 (0x8E or ESC N), SS3 (0x8F or ESC O)).  */
    /***ja
	VOVtg (SS2 (0x8E or ESC N), SS3 (0x8F or ESC O)) gB  */
    MCODING_ISO_SINGLE_SHIFT =	0x200,

    /***en
	Use 7-bit single shift 2 (SS2 (0x19)).  */
    /***ja
	7rbgVOVtg 2 (SS2 (0x19)) gB  */
    MCODING_ISO_SINGLE_SHIFT_7 =	0x400,

    /***en
	Use EUC-TW like special shifting.  */
    /***ja
	EUC-TW ̓ʂȃVtggB  */
    MCODING_ISO_EUC_TW_SHIFT =	0x800,

    /***en
	Use ISO-6429 escape sequences to indicate direction.
	Not yet implemented.  */
    /***ja
	ISO-6429 ̃GXP[vV[NGXŕwBB */
    MCODING_ISO_ISO6429 =		0x1000,

    /***en
	On encoding, if a charset has revision number, produce escape
	sequences to specify the number.  */
    /***ja
	GR[h̍ہAZbg revision number ΂
	\킷GXP[vV[NGX𐶐B        */
    MCODING_ISO_REVISION_NUMBER =	0x2000,

    /***en
	Support all ISO-2022 charsets.  */
    /***ja
	ISO-2022 ̑SWT|[gB  */
    MCODING_ISO_FULL_SUPPORT =		0x3000,

    MCODING_ISO_FLAG_MAX
  };
/*=*/

/*** @ingroup m17nConv */
/***en
    @brief Structure for a coding system of type #MCODING_TYPE_ISO_2022.

    Structure for extra information about a coding system of type
    MCODING_TYPE_ISO_2022.  */

/***ja 
    @brief #MCODING_TYPE_ISO_2022 ^CṽR[hnŕKvȕtp\.

    MCODING_TYPE_ISO_2022 ^CṽR[hnŕKvȕtpێ
    ߂̍\́B

    @latexonly \IPAlabel{MCodingInfoISO2022} @endlatexonly  */

typedef struct
{
  /***en
      Table of numbers of an ISO2022 code extension element invoked
      to each graphic plane (Graphic Left and Graphic Right).  -1
      means no code extension element is invoked to that plane.  */

  /***ja
      e}`̈ (Graphic Left  Graphic Right) ɌĂяoĂ
      AISO2022 gvf̔ԍ̃e[uB-1 ̗͂̈ɂǂ̕
      gvfĂяoĂȂƂB   */

  int initial_invocation[2];

  /***en
      Table of code extension elements.  The Nth element corresponds
      to the Nth charset in $CHARSET_NAMES, which is an argument given
      to the mconv_define_coding () function.

      If an element value is 0..3, it specifies a graphic register
      number to designate the corresponds charset.  In addition, the
      charset is initially designated to that graphic register.

      If the value is -4..-1, it specifies a graphic register number
      0..3 respectively to designate the corresponds charset.
      Initially, the charset is not designated to any graphic
      register.  */

  /***ja
      gvf̃e[uBNԖڂ̗vf́A$CHARSET_NAMES  N Ԗ
      ̕ZbgɑΉB$CHARSET_NAMES ͊֐ 
      mconv_define_coding () ̈ƂȂB

      l 0..3 AΉ镶Zbg G0..G3 ̂ꂼɎw
      邱ƁAԂłł G0..G3 ɎwĂ邱ƂӖ
      B

      l -4..-1 AΉ镶Zbg G0..G3 ̂ꂼɎw
       邪AԂł͂ǂɂwĂȂƂӖB
        */

  char designations[32];

  /***en
      Bitwise OR of @c enum @c MCodingFlagISO2022 .  */

  /***ja
      @c enum @c MCodingFlagISO2022 ̃rbgPʂł̘_ OR  */

  unsigned flags;

} MCodingInfoISO2022;
/*=*/

/*** @ingroup m17nConv */
/***en
    @brief Structure for extra information about a coding system of type #MCODING_TYPE_UTF.
    */

/***ja
    @brief #MCODING_TYPE_UTF ^CṽR[hnŕKvȕtp̍\.

    @latexonly \IPApage{MCodingInfoUTF} @endlatexonly

    @latexonly \IPAlabel{MCodingInfoUTF} @endlatexonly  */

typedef struct
{
  /***en
      Specify bits of a code unit.  The value must be 8, 16, or 32.  */
  /***ja
      R[hirbgjBl 8, 16, 32 ̂ꂩB  */
  int code_unit_bits;

  /***en
      Specify how to handle the heading BOM (byte order mark).  The
      value must be 0, 1, or 2.  The meanings are as follows:

      0: On decoding, check the first two byte.  If they are BOM,
      decide endian by them. If not, decide endian by the member @c
      endian.  On encoding, produce byte sequence according to
      @c endian with heading BOM.

      1: On decoding, do not handle the first two bytes as BOM, and
      decide endian by @c endian.  On encoding, produce byte sequence
      according to @c endian without BOM.

      2: On decoding, handle the first two bytes as BOM and decide
      ending by them.  On encoding, produce byte sequence according to
      @c endian with heading BOM.

      If <code_unit_bits> is 8, the value has no meaning.  */

  /***ja
      擪 BOM (oCgI[_[}[N) ̎舵w肷Bl 0,
      1, 2 ̂ꂩłAꂼ̈Ӗ͈ȉ̂悤ɂȂB

      0: fR[h̍ۂɍŏ 2 oCg𒲂ׂBꂪ BOM ł
      ΁AGfBAŔ肷BłȂ΁Ao @c
      endian ɏ]ăGfBA肷BGR[h̍ۂɂ @c
      endian ɏ]oCg擪 BOM tŐB

      1: fR[h̍ہAŏ 2 oCg BOM ƂĈ킸AGfBA
       @c endian Ŕ肷BGR[h̍ۂɂ́ABOM o͂A
      @c endian ɉoCg𐶐B

      2: fR[h̍ۂɍŏ2oCg BOMƂĈAɏ]ăG
      fBA𔻒肷BGR[h̍ۂɂ @c endian ɉoCg
      擪 BOM tŐB  */
  int bom;

  /***en
      Specify the endian type.  The value must be 0 or 1.  0 means
      little endian, and 1 means big endian.

      If <code_unit_bits> is 8, the value has no meaning.  */
  /***ja
      GfBÃ^Cvw肷Bl 0  1 łA0 Ȃ΃g
      GfBAA1 Ȃ΃rbOGfBAłB

      <code_unit_bits>  8 ̏ꍇɂ́A̒l͈ӖȂB
  */
  int endian;
} MCodingInfoUTF;
/*=*/

extern MSymbol mconv_define_coding (const char *name, MPlist *plist,
				    int (*resetter) (MConverter *),
				    int (*decoder) (const unsigned char *, int,
						    MText *, MConverter *),
				    int (*encoder) (MText *, int, int,
						    unsigned char *, int,
						    MConverter *),
				    void *extra_info);

extern MSymbol mconv_resolve_coding (MSymbol symbol);

extern int mconv_list_codings (MSymbol **symbols);

extern MConverter *mconv_buffer_converter (MSymbol coding, unsigned char *buf,
					   int n);

extern MConverter *mconv_stream_converter (MSymbol coding, FILE *fp);

extern int mconv_reset_converter (MConverter *converter);

extern void mconv_free_converter (MConverter *converter);

extern MConverter *mconv_rebind_buffer (MConverter *converter,
					unsigned char *buf, int n);

extern MConverter *mconv_rebind_stream (MConverter *converter, FILE *fp);

extern MText *mconv_decode (MConverter *converter, MText *mt);

MText *mconv_decode_buffer (MSymbol name, unsigned char *buf, int n);

MText *mconv_decode_stream (MSymbol name, FILE *fp);   

extern int mconv_encode (MConverter *converter, MText *mt);

extern int mconv_encode_range (MConverter *converter, MText *mt,
			       int from, int to);

extern int mconv_encode_buffer (MSymbol name, MText *mt,
				unsigned char *buf, int n);

extern int mconv_encode_stream (MSymbol name, MText *mt, FILE *fp);

extern int mconv_getc (MConverter *converter);

extern int mconv_ungetc (MConverter *converter, int c);

extern int mconv_putc (MConverter *converter, int c);

extern MText *mconv_gets (MConverter *converter, MText *mt);

/* (S4) Locale related functions corresponding to libc functions */
/*=*/
/*** @ingroup m17nShell */
/***en @defgroup m17nLocale Locale */
/***ja @defgroup m17nLocale P[ */
/*=*/

/*** @ingroup m17nLocale */
/***en
    @brief @c struct @c MLocale.

    The structure @c MLocale is used to hold information about name,
    language, territory, modifier, codeset, and the corresponding
    coding system of locales.

    The contents of this structure are implementation dependent.  Its
    internal structure is concealed from application programs.  */

/***ja
    @brief @c MLocale \.

    @c MLocale \̂́AP[̖OAAnAfBt@CAA
    R[hZbgAёΉR[hnɊւێ邽߂ɗp
    B

    ̍\̂̓e͎ɈˑB \̓AvP[Vv
    O͌ȂB  */

/***
    @seealso
    mlocale_get_prop () */

typedef struct MLocale MLocale;

/*=*/

extern MSymbol Mlanguage;
extern MSymbol Mterritory;
extern MSymbol Mmodifier;
extern MSymbol Mcodeset;

extern MLocale *mlocale_set (int category, const char *locale);

extern MSymbol mlocale_get_prop (MLocale *locale, MSymbol key);

extern int mtext_ftime (MText *mt, const char *format, const struct tm *tm,
			MLocale *locale);

extern MText *mtext_getenv (const char *name);

extern int mtext_putenv (MText *mt);

extern int mtext_coll (MText *mt1, MText *mt2);

/*
 *  (9) Miscellaneous functions of libc level (not yet implemented)
 */

/*
extern int mtext_width (MText *mt, int n);
extern MText *mtext_tolower (MText *mt);
extern MText *mtext_toupper (MText *mt);
*/

/*
 *  (10) Input method
 */
/*=*/
/*** @ingroup m17nShell */
/***en @defgroup m17nInputMethod Input Method (basic) */
/***ja @defgroup m17nInputMethod ̓\bh ({) */
/*=*/

/*** @addtogroup m17nInputMethod 
     @{ */

/* Struct forward declaration.  */
typedef struct MInputMethod MInputMethod;
typedef struct MInputContext MInputContext;

/***en
    @brief Type of input method callback functions.

    This is the type of callback functions called from input method
    drivers.  #IC is a pointer to an input context, #COMMAND is a name
    of callback for which the function is called.   */
/***ja
    @brief ̓\bhR[obN֐̌^錾.

    ̓\bhĂ΂R[obN֐̌^łB#IC ͓̓R
    eNXgւ̃|C^A#COMMAND ͊֐Ă΂R[obN̖O
    łB  */

typedef void (*MInputCallbackFunc) (MInputContext *ic, MSymbol command);
/*=*/

/***en
    @brief Structure of input method driver.

    The type @c MInputDriver is the structure of an input method driver that
    contains several functions to handle an input method.  */

/***ja
    @brief ̓hCop\.

    @c MInputDriver ́A̓\bh舵֐܂ޓ̓\bhh
    Co̍\̂̌^łB  */

typedef struct MInputDriver
{
  /***en
      @brief Open an input method.

      This function opens the input method $IM.  It is called from the
      function minput_open_im () after all member of $IM but <info>
      set.  If opening $IM succeeds, it returns 0.  Otherwise, it
      returns -1.  The function can setup $IM->info to keep various
      information that is referred by the other driver functions.  */

  /***ja
      @brief ̓\bhI[v.

      ̊֐́A̓\bh $IM I[vB$IM  <info> ȊO
      ̑So[ZbgꂽŁA֐ minput_open_im () Ă
      B$IM I[vł 0 AłȂ -1ԂB̊
       $IM->info ݒ肵āÃhCo֐QƂ
      ێ邱ƂłB
      */

  int (*open_im) (MInputMethod *im);

  /***en
      @brief Close an input method.

      This function closes the input method $IM.  It is called from
      the function minput_close_im ().  It frees all memory allocated
      for $IM->info (if any) after finishing all the tasks of closing
      the input method.  But, the other members of $IM should not be
      touched.  */

  /***ja
      @brief ̓\bhN[Y.

      ̊֐́A̓\bh $IM N[YB֐ 
      minput_close_im () Ă΂B̓\bh̃N[Yׂ
      I_ŁA$IM->info Ɋ蓖ĂĂ郁()
      ׂĊJBA$IM ̑̃oɉe^Ă͂ȂȂB
        */

  void (*close_im) (MInputMethod *im);

  /***en
      @brief Create an input context.

      This function creates the input context $IC.  It is called from
      the function minput_create_ic () after all members of $IC but
      <info> are set.  If creating $IC succeeds, it returns 0.
      Otherwise, it returns -1.  The function can setup $IC->info to
      keep various information that is referred by the other driver
      functions.  */

  /***ja
      @brief ̓ReNXg𐶐.

      ̊֐͓̓ReNXg $IC 𐶐B$IC  <info> ȊO
      So[ZbgꂽŁA֐ minput_create_ic () Ă
      B$IC 𐶐ł 0 AłȂ -1 ԂB̊֐
       $IC->info ݒ肵āÃhCo֐QƂ
      邱ƂłB  */


  int (*create_ic) (MInputContext *ic);

  /***en
      @brief Destroy an input context.

      This function is called from the function minput_destroy_ic ()
      and destroys the input context $IC.  It frees all memory
      allocated for $IC->info (if any) after finishing all the tasks
      of destroying the input method.  But, the other members of $IC
      should not be touched.  */

  /***ja
      @brief ̓ReNXgj󂷂.

      ֐ minput_destroy_ic () Ă΂A̓ReNXg $IC j
      󂷂B̓ReNXg̔j󂪂ׂďI_ŁA$IC->info 
      Ɋ蓖ĂĂ郁()ׂĊJBA$IC 
      ̑̃oɉe^Ă͂ȂȂB  */

  void (*destroy_ic) (MInputContext *ic);

  /***en
      @brief Filter an input key.

      This function is called from the function minput_filter () and
      filters an input key.  $KEY and $ARG are the same as what given
      to minput_filter ().

      The task of the function is to handle $KEY, update the internal
      state of $IC.  If $KEY is absorbed by the input method and no
      text is produced, it returns 1.  Otherwise, it returns 0.

      It may update $IC->status, $IC->preedit, $IC->cursor_pos,
      $IC->ncandidates, $IC->candidates, and $IC->produced if that is
      necessary for the member <callback>.

      The meaning of $ARG depends on the input method river.  See the
      documentation of @c minput_default_driver and @c
      minput_gui_driver for instance.  */

  /***ja
      @brief ̓L[tB^.

      ֐ minput_filter () Ă΂A̓L[tB^B 
      $KEY, $ARG ͊֐ minput_filter () ̂̂ƓB

      ̊֐ $KEY A$IC ̓ԂXVB $KEY 
      \bhɋzăeLXgȂꍇɂ́A 1 
      BłȂ 0 ԂB

      o <callback> ɕKvł΁A$IC->status, $IC->preedit,
      $IC->cursor_pos, $IC->ncandidates, $IC->candidates,
      $IC->produced XVłB

      $ARG ̈Ӗ͓̓\bhhCoɈˑB @c
      minput_default_driver ܂ @c minput_gui_driver ̐QƂ
      ƁB */

  int (*filter) (MInputContext *ic, MSymbol key, void *arg);

  /***en
      @brief  Lookup a produced text in an input context.

      It is called from the function minput_lookup () and looks up a
      produced text in the input context $IC.  This function
      concatenate a text produced by the input key $KEY (if any) to
      M-text $MT.  If $KEY was correctly handled by the input method
      of $IC, it returns 0.  Otherwise, it returns 1.

      The meaning of $ARG depends on the input method driver.  See the
      documentation of @c minput_default_driver and @c
      minput_gui_driver for instance.  */

  /***ja
      @brief ̓ReNXgŐeLXg̊l.

      ֐ minput_lookup () Ă΂A̓ReNXg $IC Ő
      eLXgB̓L[ $KEY ɂĐeLXg
      ΁AM-text $MT ɒǉB $KEY ̓\bh $IC ɂ
      Đ 0 ԂBłȂ 1 ԂB

      $ARG ̈Ӗ͓̓\bhhCoɈˑB @c
      minput_default_driver ܂ @c minput_gui_driver ̐QƂ
      ƁB */

  int (*lookup) (MInputContext *ic, MSymbol key, void *arg, MText *mt);

  /***en
      @brief List of callback functions.

      List of callback functions.  Keys are one of
      #Minput_preedit_start, #Minput_preedit_draw,
      #Minput_preedit_done, #Minput_status_start, #Minput_status_draw,
      #Minput_status_done, #Minput_candidates_start,
      #Minput_candidates_draw, #Minput_candidates_done,
      #Minput_set_spot, #Minput_toggle, #Minput_reset.
      Values are functions of type #MInputCallbackFunc.  */
  /***ja
      @brief R[obN֐̃Xg.

      R[obN֐̃XgBL[͎̂ꂩB
      #Minput_preedit_start, #Minput_preedit_draw,
      #Minput_preedit_done, #Minput_status_start, #Minput_status_draw,
      #Minput_status_done, #Minput_candidates_start,
      #Minput_candidates_draw, #Minput_candidates_done,
      #Minput_set_spot, #Minput_toggle, #Minput_resetB 
      l#MInputCallbackFunc ^̊֐B  */
  MPlist *callback_list;

} MInputDriver;
/*=*/

extern MInputDriver minput_default_driver;

extern MSymbol Minput_driver;

extern MInputDriver *minput_driver;

/** Symbols for callback commands.  */
extern MSymbol Minput_preedit_start;
extern MSymbol Minput_preedit_draw;
extern MSymbol Minput_preedit_done;
extern MSymbol Minput_status_start;
extern MSymbol Minput_status_draw;
extern MSymbol Minput_status_done;
extern MSymbol Minput_candidates_start;
extern MSymbol Minput_candidates_draw;
extern MSymbol Minput_candidates_done;
extern MSymbol Minput_set_spot;
extern MSymbol Minput_toggle;
extern MSymbol Minput_reset;

/***en
    @brief Structure of input method.

    The type @c MInputMethod is the structure of input method
    objects.  */
/***ja
    @brief ̓\bh̍\.

    @c MInputMethod ́A̓\bhIuWFNgp̍\̂̌^łB  */

struct MInputMethod
{
  /***en Which language this input method is for.  The value is @c
      Mnil if the input method is foreign.  */
  /***ja ǂ̌p̓̓\bhB̓\bhÔ̂ł
      ɂ͒lƂ @c Mnil B  */
  MSymbol language;

  /***en Name of the input method.  If the input method is foreign, it
      must has a property of key @c Minput_driver and the value must be a
      pointer to a proper input method driver.  */
  /***ja ̓\bh̖OBO\bhłꍇɂ́A@c
      Minput_driver L[ƂvpeBA̒l͓K؂ȓ̓
      \bhhCoւ̃|C^łȂĂ͂ȂȂB*/
  MSymbol name;

  /***en Input method driver of the input method.  */
  /***ja ̓̓\bhp̓̓\bhhCoB  */
  MInputDriver driver;

  /***en The argument given to minput_open_im (). */
  /***ja minput_open_im () ɓnB  */
  void *arg;

  /***en Pointer to extra information that <driver>.open_im ()
      setups. */
  /***ja <driver>.open_im () ݒ肷ǉւ̃|C^B */
  void *info;
};

/*=*/

/***en
    @brief Structure of input context.

    The type @c MInputContext is the structure of input context
    objects.  */

/***ja
    @brief ̓ReNXgp\.

    @c MInputContext ́A̓ReNXgIuWFNgp̍\̂̌^
    B  */

struct MInputContext
{
  /***en Backward pointer to the input method.  It is set up be the
      function minput_create_ic ().  */
  /***ja ̓\bhւ̋t|C^B֐ minput_create_ic () ɂ
      ݒ肳B  */ 
  MInputMethod *im;

  /***en M-text produced by the input method.  It is set up by the
      function minput_lookup () .  */
  /***ja ̓\bhɂĐ M-textB֐ minput_lookup () 
      ɂĐݒ肳B  */
  MText *produced;

  /***en Argument given to the function minput_create_im (). */
  /***ja ֐ minput_create_ic () ɓnB */
  void *arg;

  /***en Flag telling whether the input context is currently active or
      inactive.  The value is set to 1 (active) when the input context
      is created.  It can be toggled by the function minput_toggle
      ().  */
  /***ja ̓ReNXgANeBuǂtOB̓Re
      NXgꂽ_ł͒l 1 iANeBujłA֐ 
      minput_toggle () ɂăgOB  */
  int active;


  /***en Spot location and size of the input context.  */
  /***ja ̓ReNXg̃X|bg̈ʒuƑ傫.  */
  struct {
    /***en X and Y coordinate of the spot.  */
    /***ja X|bg X, Y W.  */
    int x, y;

    /***en Ascent and descent pixels of the line of the spot.  */
    /***ja X|bg̃AZgƃfBZg̃sNZ.  */
    int ascent, descent;

    /***en Font size for preedit text in 1/10 point.  */
    /***ja preedit eLXgp̃tHgTCY (1/10 |CgP).  */
    int fontsize;

    /***en M-text at the spot, or NULL.  */
    /***ja X|bg M-textA܂ NULL.  */
    MText *mt;

    /***en Character position in <mt> at the spot.  */
    /***ja <mt> ɂX|bg̕ʒu.  */
    int pos;
  } spot;

  /***en The usage of the following members depends on the input
      method driver.  The descriptions below are for the driver of an
      internal input method.  They are set by the function
      <im>->driver.filter ().  */
  /***ja ȉ̃o̎gp@͓̓\bhhCoɂĈقȂB
      ̐́A̓\bhp̓̓hCoɑ΂̂łB
      ͊֐ <im>->driver.filter () ɂĐݒ肳B  */

  /***en Pointer to extra information that <im>->driver.create_ic ()
      setups.  It is used to record the internal state of the input
      context.  */
  /***ja <im>->driver.create_ic () ݒ肷ǉւ̃|C^B
      ̓ReNXg̓ԂL^邽߂ɗpB */
  void *info;

  /***en M-text describing the current status of the input
      context.  */
  /***ja ̓ReNXǧ݂̏󋵂\ M-text  */
  MText *status;

  /***en The function <im>->driver.filter () sets the value to 1 when
      it changes <status>.  */
  /***ja ֐ <im>->driver.filter () ́A<status> ς̒l 1 
      ɐݒ肷B  */
  int status_changed;

  /***en M-text containing the current preedit text.  The function
      <im>->driver.filter () sets the value.  */
  /***ja ݂ preedit eLXg܂ M-textB֐ 
      <im>->driver.filter () ̒lݒ肷B  */ 
  MText *preedit;

  /***en The function <im>->driver.filter () sets the value to 1 when
      it changes <preedit>.  */
  /***ja ֐ <im>->driver.filter () ́A<preedit> ς̒l 
      1 ɐݒ肷B  */
  int preedit_changed;

  /***en Cursor position of <preedit>.  */
  /***ja <preedit>̃J[\ʒu  */
  int cursor_pos;

  int cursor_pos_changed;

  /***en Array of the current candidate texts.  */
  /***ja ݂̃eLXg̔z.  */
  MPlist *candidate_list;
  int candidate_index;
  int candidate_from, candidate_to;
  int candidate_show;

  /***en The function <im>->driver.filter () sets the value to 1 when
      it changes one of the above members.  */
  /***ja ֐ <im>->driver.filter () ́AL̃ôPς
      ̒l 1 ɐݒ肷B  */
  int candidates_changed;

  MPlist *plist;
};

/*=*/

extern MInputMethod *minput_open_im (MSymbol language, MSymbol name,
				     void *arg);

/*=*/

extern void minput_close_im (MInputMethod *im);

extern MInputContext *minput_create_ic (MInputMethod *im, void *arg);

extern void minput_destroy_ic (MInputContext *ic);

extern int minput_filter (MInputContext *ic, MSymbol key, void *arg);

extern int minput_lookup (MInputContext *ic, MSymbol key, void *arg,
			  MText *mt);

extern void minput_set_spot (MInputContext *ic, int x, int y,
			     int ascent, int descent, int fontsize,
			     MText *mt, int pos);

extern void minput_toggle (MInputContext *ic);

extern void minput_reset_ic (MInputContext *ic);
/*=*/
/***     @} */

extern MInputMethod *mdebug_dump_im (MInputMethod *im, int indent);


#ifdef __cplusplus
}
#endif

#endif /* _M17N_H_ */

/*
  Local Variables:
  coding: euc-japan
  End:
*/
